
import java.awt.*;
import java.awt.event.*;
import java.util.*;

public class Boids extends Canvas {

    
    public Boids (int width, int height, Image weightControlImage) {

	SIZE = new Dimension(width, height);

	weightControl = new WeightControl(weightControlImage);

	buffer = createImage(width, height);

	initBoids();
	target.x = target.y = 100;


	BoidsMouseListener listener = new BoidsMouseListener();

	addMouseListener(listener);
	addMouseMotionListener(listener);
    }

    public synchronized void setTarget(Point p) {
	target = p;
    }

    public void calculateAcceleration(DoublePoint acceleration, int index) {

	// First, the target influence
	DoublePoint targetInfluence = new DoublePoint();
	targetInfluence.x = target.x - position[index].x;
	targetInfluence.y = target.y - position[index].y;
	normalizeVector(targetInfluence);
	lengthenVector(targetInfluence, maxAcceleration);
		
	DoublePoint distance[] = new DoublePoint[BOID_COUNT];
	DoublePoint normalDistance[] = new DoublePoint[BOID_COUNT];
	double length[] = new double[BOID_COUNT];
	double minLength = Integer.MAX_VALUE;
	int closestIndex = 0;
	for (int i=0; i<position.length; i++) {
	    distance[i] = new DoublePoint();
	    normalDistance[i] = new DoublePoint();
	    distance[i].x = normalDistance[i].x = position[i].x - position[index].x;
	    distance[i].y = normalDistance[i].y = position[i].y - position[index].y;
	    length[i] = vectorLength(distance[i]);	    
	    if ((length[i] < minLength) && (i != index)) {
		closestIndex = i;
		minLength = length[i];
	    }
	    normalizeVector(normalDistance[i]);
	}

	// Next, the cluster influence
	DoublePoint clusterInfluence = new DoublePoint();
	double radius = 50;
	double clusterThreshold = 50;
	for (int i=0; i<position.length; i++) {
	    if (length[i] < clusterThreshold) {
		clusterInfluence.x += maxAcceleration*normalDistance[i].x;
		clusterInfluence.y += maxAcceleration*normalDistance[i].y;
	    }
	}
	
	truncateVector(clusterInfluence, maxAcceleration);
	
	// Next, the dispersion influence
	DoublePoint dispersionInfluence = new DoublePoint();
	double dispersionThreshold = 20;
	for (int i=0; i<position.length; i++) {
	    if ((length[i] < dispersionThreshold) && (length[i] > 0)) {
		dispersionInfluence.x -= maxAcceleration*normalDistance[i].x*(dispersionThreshold-length[i])*(dispersionThreshold-length[i]);
		dispersionInfluence.y -= maxAcceleration*normalDistance[i].y*(dispersionThreshold-length[i])*(dispersionThreshold-length[i]);
		clusterInfluence.x = targetInfluence.x = 0;
		clusterInfluence.y = targetInfluence.y = 0;
	    }
	}
	truncateVector(dispersionInfluence, maxAcceleration);
	
	// Next, the avoid-the-obstacle influence
	Obstacle o;
	double dangerRadius = 100;
	DoublePoint dangerZone = new DoublePoint(orientation[index].x, orientation[index].y);	
	double intersections[];
	DoublePoint obstacleInfluence = new DoublePoint();
	DoublePoint obstacleDistance = new DoublePoint();
	lengthenVector(dangerZone, dangerRadius);

	for (int i=0; i<obstacles.size(); i++) {
	    o = (Obstacle)obstacles.elementAt(i);
	    intersections = o.intersections(position[index].x, position[index].y, 
					    (int)Math.round(position[index].x + dangerZone.x),
					    (int)Math.round(position[index].y + dangerZone.y));

	    if ((Math.abs(intersections[0]) <= 1.0) || (Math.abs(intersections[1]) <= 1.0)) {
		//		System.out.println("I'm gonna crash!: ["+intersections[0]+", "+intersections[1]+"]");
		obstacleDistance.x = o.center.x - position[index].x;
		obstacleDistance.y = o.center.y - position[index].y;
		normalizeVector(obstacleDistance);
		obstacleInfluence.x -= maxAcceleration*obstacleDistance.x;
		obstacleInfluence.y -= maxAcceleration*obstacleDistance.y;
		//		clusterInfluence.x = dispersionInfluence.x = 0;
		//		clusterInfluence.y = dispersionInfluence.y = 0;
	    }
	}

	DoublePoint orientationInfluence = new DoublePoint();
	double orientationThreshold = 20;
	for (int i=0; i<orientation.length; i++) {
	    if (length[i] < orientationThreshold) {
		orientationInfluence.x += orientation[i].x;
		orientationInfluence.y += orientation[i].y;
	    }
	}
	normalizeVector(orientationInfluence);
	lengthenVector(orientationInfluence, maxAcceleration);

	// Combine the weighted influences

	acceleration.x = acceleration.y = 0;

	acceleration.x += TARGET_WEIGHT * targetInfluence.x;
	acceleration.y += TARGET_WEIGHT * targetInfluence.y;

	acceleration.x += CLUSTER_WEIGHT * clusterInfluence.x;
	acceleration.y += CLUSTER_WEIGHT * clusterInfluence.y;

	acceleration.x += DISPERSION_WEIGHT * dispersionInfluence.x;
	acceleration.y += DISPERSION_WEIGHT * dispersionInfluence.y;

	acceleration.x += OBSTACLE_WEIGHT * obstacleInfluence.x;
	acceleration.y += OBSTACLE_WEIGHT * obstacleInfluence.y;

	acceleration.x += ORIENTATION_WEIGHT * orientationInfluence.x;
	acceleration.y += ORIENTATION_WEIGHT * orientationInfluence.y;
    }
    
    public synchronized void step() {
		
	for (int i=0; i<BOID_COUNT; i++) {
	    position[i].x += Math.round(velocity[i].x);
	    position[i].y += Math.round(velocity[i].y);	    
	    if (position[i].x < 0)
		position[i].x += SIZE.width;
	    else
		if (position[i].x > SIZE.width)
		    position[i].x = position[i].x % SIZE.width;

	    if (position[i].y < 0)
		position[i].y += SIZE.height;
	    else
		if (position[i].y > SIZE.height)
		    position[i].y = position[i].y % SIZE.height;
	}
	for (int i=0; i<BOID_COUNT; i++) {
	    calculateAcceleration(acceleration[i], i);
	
	    truncateVector(acceleration[i], maxAcceleration);
	    lengthenVector(acceleration[i], maxAcceleration);
	}

	for (int i=0; i<BOID_COUNT; i++) {
	    velocity[i].x = orientation[i].x = velocity[i].x + acceleration[i].x;
	    velocity[i].y = orientation[i].y = velocity[i].y + acceleration[i].y;
	    normalizeVector(orientation[i]);
	
	    truncateVector(velocity[i], maxVelocity);

	    Rectangle bounds = null;
	    if (shape != null)
		bounds = shape[i].getBounds();
	    recalculateShape(i);

	    if (bounds != null)
		bounds = bounds.union(shape[i].getBounds());
	    else
		bounds = shape[i].getBounds();
	
	}
	
	int i=0; 
	Obstacle o;
	while (i<obstacles.size()) {
	    o = (Obstacle)obstacles.elementAt(i);
	    if (o.radius < OBSTACLE_CUTOFF_RADIUS) {
		o.radius++;
	    } 
	    i++;
	}

	repaint();
    }

    private void recalculateShape(int index) {
	double x, y;
	for (int i=0; i < shapex.length; i++) {
	    x = 5 * (shapex[i] * orientation[index].x - shapey[i] * orientation[index].y);
	    y = 5 * (shapex[i] * orientation[index].y + shapey[i] * orientation[index].x);
	    shape[index].xpoints[i] = (int) Math.round(x);
	    shape[index].ypoints[i] = (int) Math.round(y);
	}
	shape[index].translate(position[index].x, position[index].y);
    }
    
    public void update(Graphics g) {
	paint(g);
    }

    public void paint(Graphics g) {

	Graphics bufferG;

	if (buffer == null)
	    buffer = getParent().createImage(SIZE.width, SIZE.height);

	bufferG = buffer.getGraphics();

	Rectangle bounds = getBounds();
	bufferG.setColor(Color.white);
	bufferG.fillRect(0, 0, SIZE.width, SIZE.height);

	for (int i=0; i<shape.length; i++) 
	    if (shape[i] != null) {
		bufferG.setColor(Color.red);
		bufferG.fillPolygon(shape[i]);
		bufferG.setColor(Color.black);
		bufferG.drawPolygon(shape[i]);
	    }
	
	bufferG.setColor(Color.green);
	bufferG.drawLine(target.x-10, target.y, target.x+10, target.y);
	bufferG.drawLine(target.x, target.y-10, target.x, target.y+10);
	bufferG.setColor(Color.black);
	bufferG.drawLine(target.x-9, target.y+1, target.x+11, target.y+1);
	bufferG.drawLine(target.x+1, target.y-9, target.x+1, target.y+11);
	
	synchronized(this) {
	    Obstacle o;
	    bufferG.setColor(Color.blue);
	    for (int i=0; i<obstacles.size(); i++) {
		o = (Obstacle)obstacles.elementAt(i);
		bufferG.drawOval((int)Math.round(o.center.x-o.radius),
			   (int)Math.round(o.center.y-o.radius),
			   (int)Math.round(2*o.radius), 
			   (int)Math.round(2*o.radius));
	    }
	}

	Dimension size = weightControl.getPreferredSize();
	Point delta = new Point(bounds.width - size.width,
				bounds.height - size.height);
	bufferG.translate(delta.x, delta.y);
	weightControl.paint(bufferG);
	bufferG.translate(-delta.x, -delta.y);

	bufferG.setColor(Color.black);
	bufferG.drawRect(0, 0, bounds.width-1, bounds.height-1);

	g.drawImage(buffer, 0, 0, this);
    }

    public Dimension getPreferredSize() {
	return SIZE;
    }

    void initBoids() {

	acceleration = new DoublePoint[BOID_COUNT];
	velocity = new DoublePoint[BOID_COUNT];
	orientation = new DoublePoint[BOID_COUNT];
	shape = new Polygon[BOID_COUNT];
	position = new Point[BOID_COUNT];
	
	Random random = new Random();
	for (int i=0; i<BOID_COUNT; i++) {

	    acceleration[i] = new DoublePoint();

	    velocity[i] = new DoublePoint();
	    velocity[i].x = Math.round((random.nextFloat() - .5) * maxVelocity * 2);
	    velocity[i].y = Math.round((random.nextFloat() - .5) * maxVelocity * 2);
	    truncateVector(velocity[i], maxVelocity);
	    
	    orientation[i] = new DoublePoint();
	    orientation[i].x = velocity[i].x;
	    orientation[i].y = velocity[i].y;
	    normalizeVector(orientation[i]);
	    
	    shape[i] = new Polygon();
	    for (int j=0; j<shapex.length; j++)
		shape[i].addPoint(0, 0);

	    position[i] = new Point();
	    position[i].x = Math.round(random.nextFloat() * SIZE.width);
	    position[i].y = Math.round(random.nextFloat() * SIZE.height);
	}
    }
    
	
    static double pixelDistance(Point a, Point b) {
	return Math.sqrt((b.x - a.x)*(b.x - a.x) + (b.y - a.y)*(b.y - a.y));
    }

    static double vectorLength(DoublePoint vector) {
        return Math.sqrt(vector.x*vector.x + vector.y*vector.y);
    }

    static double vectorLengthSquared(DoublePoint vector) {
        return vector.x*vector.x + vector.y*vector.y;
    }

    static void normalizeVector(DoublePoint vector) {
        double length = vectorLength(vector);
        if (length == 0) return;

        vector.x /= length;
        vector.y /= length;
    }

    static void truncateVector(DoublePoint vector, double maxVal) {
        if (vectorLengthSquared(vector) > maxVal*maxVal) {
            normalizeVector(vector);
            vector.x *= maxVal;
            vector.y *= maxVal;
        }
    }

    static void lengthenVector(DoublePoint vector, double val) {
        double length = vectorLength(vector);
        if (length == 0) return;

        double increase = (length + val) / length;
        vector.x *= increase;
        vector.y *= increase;
    }

    static void divideVector(DoublePoint vector, double val) {
        vector.x /= val;
        vector.y /= val;
    }


    private class BoidsMouseListener extends MouseAdapter implements MouseMotionListener {
	public void mousePressed(MouseEvent event) {			
	    mousePressedModifiers = event.getModifiers();
	    if (eventShouldBeDispatched(event))
		forwardEvent(event);
	    else
		if ((event.getModifiers() & event.BUTTON1_MASK) != 0) {
		    if (event.isShiftDown()) {
			// nothin' yet
		    }
		    else
			if (event.isControlDown()) { 
			    obstacles.addElement(new Obstacle(event.getX(), 
							      event.getY(), 
							      1));
			}
			else {
			    setTarget(event.getPoint());
			}
		}
		else {
				// nothin' yet
		}
	}

	public void mouseDragged(MouseEvent event) {

	    if (eventShouldBeDispatched(event))
		forwardEvent(event);
	    else
		if ((mousePressedModifiers & event.BUTTON1_MASK) != 0) {
		    if (event.isShiftDown()) {
			// nothin' yet
		    }
		    else
			if (event.isControlDown()) {
			    // nothin' yet
			}
			else {
			    setTarget(event.getPoint());
			}
		}
		else {
				// nothin' yet
		}
	}
	public void mouseMoved(MouseEvent event) {
	}

	private boolean eventShouldBeDispatched(MouseEvent event) {
	    Rectangle bounds = getBounds();
	    Dimension size = weightControl.getPreferredSize();
	    return ((event.getX() > (bounds.width - size.width)) &&
		    (event.getY() > (bounds.height - size.height)));
	}
	
	private void forwardEvent(MouseEvent event) {
	    Rectangle bounds = getBounds();
	    Dimension size = weightControl.getPreferredSize();
	    Point translate = new Point(bounds.width - size.width, 
					bounds.height - size.height);
	    event.translatePoint(-translate.x,-translate.y);
	    weightControl.dispatchEvent(event);
	}

	// need this so that the mouseDragged event can process it
	private int mousePressedModifiers = 0;
    }

    Point target = new Point();

    Vector obstacles = new Vector();

    double maxAcceleration = 2;
    double maxVelocity = 10;

    Point[] position;
    DoublePoint acceleration[];
    DoublePoint[] velocity;
    DoublePoint[] orientation;
    Polygon[] shape;

    WeightControl weightControl;


    Image buffer;

    int BOID_COUNT = 50;
    Dimension SIZE = new Dimension(320, 400);
    int OBSTACLE_CUTOFF_RADIUS = 50;

    static double TARGET_WEIGHT = 1;
    static double CLUSTER_WEIGHT = 1;
    static double DISPERSION_WEIGHT = 1;
    static double OBSTACLE_WEIGHT = 5;
    static double ORIENTATION_WEIGHT = 1;
    
    static final float shapex[] = { -1f, -1.5f, 1f, -1.5f };
    static final float shapey[] = {  0f, -1f,   0f,  1f };
}
